<?php
// ✅ WordPress 패널 <-> Google Apps Script 완전 연동 (최종 안정판 2025.11)
// -----------------------------------------------------------
// 기능 요약
// ① GPT/Gemini API 키 저장
// ② 도메인 계정 추가 / 수정 / 삭제
// ③ Google Apps Script ScriptProperties 자동 동기화
// -----------------------------------------------------------

header('Content-Type: application/json; charset=utf-8');

// -----------------------------------------------------------
// [1] 로컬 accounts.json 초기화
// -----------------------------------------------------------
$file = __DIR__ . '/accounts.json';
if (!file_exists($file) || filesize($file) === 0) {
    file_put_contents($file, json_encode([
        "api" => ["gpt_api" => "", "gemini_api" => ""],
        "domains" => []
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

$json = json_decode(file_get_contents($file), true);
if (!$json) $json = ["api" => ["gpt_api" => "", "gemini_api" => ""], "domains" => []];

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'load':
        echo json_encode($json, JSON_UNESCAPED_UNICODE);
        break;

    case 'save_api':
        $input = json_decode(file_get_contents('php://input'), true);
        $json['api']['gpt_api'] = trim($input['gpt_api'] ?? '');
        $json['api']['gemini_api'] = trim($input['gemini_api'] ?? '');
        file_put_contents($file, json_encode($json, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        // API 키 변경 시 동기화 실행
        sendToGoogleAppsScript($json);
        
        echo json_encode(["status" => "ok"]);
        break;

    case 'add':
    case 'update':
    case 'delete':
        handleAccountAction($action, $json, $file);
        break;

    default:
        echo json_encode(["error" => "invalid_action"]);
        break;
}


// ============================================================
// ✅ [1] 계정 추가/수정/삭제 + GAS 동기화
// ============================================================
function handleAccountAction($action, &$json, $file)
{
    $input = json_decode(file_get_contents('php://input'), true);

    if ($action === 'add') {
        $domain = trim($input['domain'] ?? '');
        $username = trim($input['username'] ?? '');
        $password = trim($input['password'] ?? '');
        if (!$domain || !$username || !$password) {
            echo json_encode(["status" => "error", "message" => "모든 필드를 입력하세요."]);
            return;
        }

        foreach ($json['domains'] as $d) {
            if (strcasecmp($d['domain'], $domain) === 0) {
                echo json_encode(["status" => "duplicate"]);
                return;
            }
        }

        $json['domains'][] = ["domain" => $domain, "username" => $username, "password" => $password];
    }

    if ($action === 'update') {
        $i = intval($input['index'] ?? -1);
        if ($i < 0 || !isset($json['domains'][$i])) {
            echo json_encode(["status" => "error", "message" => "대상 없음"]);
            return;
        }
        $json['domains'][$i] = [
            "domain" => trim($input['domain']),
            "username" => trim($input['username']),
            "password" => trim($input['password'])
        ];
    }

    if ($action === 'delete') {
        $i = intval($input['index'] ?? -1);
        if ($i >= 0 && isset($json['domains'][$i])) array_splice($json['domains'], $i, 1);
    }

    // 파일 저장
    file_put_contents($file, json_encode($json, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

    // ✅ Google Apps Script 동기화
    sendToGoogleAppsScript($json);

    echo json_encode(["status" => "ok"]);
}


// ============================================================
// ✅ [2] Google Apps Script에 JSON 전송 (최종 안정화 버전)
// ============================================================
function sendToGoogleAppsScript($data)
{
    if (empty($data)) return;

    $payload = [
        "mode" => "sync_sheet",
        "api" => [
            "gpt_api" => $data['api']['gpt_api'] ?? '',
            "gemini_api" => $data['api']['gemini_api'] ?? ''
        ],
        // GAS로 전송할 도메인 데이터 (배열 형태)
        "domains" => $data['domains'] 
    ];

    $GAS_URL = "https://script.google.com/macros/s/AKfycbxrmzKl3tqWF5OcG5jrhRpRYHiKlZUl4rg6sYP4slpITJfCzGgIzsjSYpx1hdqNLs-qkw/exec";

    // 🚨 핵심 수정: JSON 데이터를 'data' 파라미터에 담아 전송
    $postData = http_build_query([
        'data' => json_encode($payload, JSON_UNESCAPED_UNICODE)
    ]);

    $ch = curl_init($GAS_URL);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 15,
        // Content-Type: application/x-www-form-urlencoded
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/x-www-form-urlencoded'
        ],
        CURLOPT_POSTFIELDS => $postData
    ]);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $log = sprintf("[%s] HTTP %s | %d domains | Response: %s\n",
        date('Y-m-d H:i:s'),
        $httpCode,
        count($data['domains']),
        trim($response ?: 'No response')
    );

    file_put_contents(__DIR__ . '/gas_debug.log', $log, FILE_APPEND);
}


// ============================================================
// ❌ [3] 도메인 키 변환 함수 삭제
// ============================================================
// 이 함수는 GAS에서 처리하도록 로직이 변경되어 삭제되었습니다.
// function convertDomainFormat($domains) { ... }
?>