/* ======================================================================
 * 📌 GPT Auto Publisher Panel (v3.2 - 최종 완성본)
 * - v3.1과 동일 (수정 중 초기화 방지, 날짜 저장 버그 수정, 발행갯수 자동 계산)
 * - 백엔드 v3.2와 완벽 호환
 * ====================================================================== */
document.addEventListener("DOMContentLoaded", () => {
  
  // ----------------------------------------------------------------------
  // [UTIL] 보조 함수
  // ----------------------------------------------------------------------

  /**
   * 시트에서 온 값 (문자열, 배열)을 일관된 배열로 변환
   */
  function parseArray(value) {
    if (!value) return [];
    try {
      if (Array.isArray(value)) return value;
      if (typeof value === "string") {
        value = value.trim();
        if (value.startsWith("[") && value.endsWith("]")) {
          return JSON.parse(value);
        } else if (value.includes(",")) {
          return value.split(",").map(v => v.trim()).filter(v => v);
        } else {
          return [value];
        }
      }
      return [];
    } catch (e) {
      console.warn("⚠️ parseArray 변환 실패:", e.message);
      return [];
    }
  }

  // ----------------------------------------------------------------------
  // [CONFIG] 전역 변수 및 상수
  // ----------------------------------------------------------------------
  
  const addRowBtn = document.getElementById("addRowBtn");
  const tbody = document.getElementById("postBody");
  const domainModal = document.getElementById("domainModal");
  const linkModal = document.getElementById("linkModal");
  const loader = document.getElementById("loader");

  let posts = []; // 메인 데이터
  let publishLogs = []; // 로그 데이터
  let currentPostIndex = -1; // 모달용 인덱스

   // ✅ [중요] 본인의 GAS 배포 URL로 변경하세요.
  const SHEET_URL = "https://script.google.com/macros/s/AKfycbzwE9_GFelQ7T-1atFf36EFk4wH0_aUsp0JKhwXUammeN0welF9hoGQMc83Uw9KTNqEPw/exec";

// ✅ 중복 요청 방지용 플래그 추가 (시트 자동 새로고침 중복 차단)
let isLoading = false;
  /* ===============================
     📥 [STEP 1] 시트 불러오기 (메인)
   =============================== */
async function loadFromSheet() {
  // ✅ 중복 호출 방지 플래그 — 이미 로드 중이면 실행 차단
  if (isLoading) {
    console.warn("⚠️ 이미 시트 로드 중입니다. 중복 요청 차단됨.");
    return;
  }
  isLoading = true;

  if (posts.length === 0) {
    tbody.innerHTML = `<tr><td colspan="18" class="p-4 text-gray-400">시트 불러오는 중...</td></tr>`;
  }

  try {
      const callbackName = "cb_" + Date.now() + "_" + Math.floor(Math.random() * 1000);

      window[callbackName] = (data) => {
        console.log("✅ 시트 데이터 로드 성공:", data);
        
        // =================================================================
        // ✅ [v3.1] 수정 중 초기화 방지 (v3.2 유지)
        // =================================================================
        // 사용자가 셀을 편집 중이거나 "저장" 버튼(💾)을 누른 상태일 때,
        // 10초 자동 새로고침이 화면을 덮어쓰는 것을 방지합니다.
const isEditing =
  document.querySelector("[contenteditable=true]") ||
  Array.from(document.querySelectorAll(".edit-btn")).some(btn => btn.innerText.includes("💾"));

        if (isEditing) {
          console.warn("⚠️ 수정 중... 렌더링을 건너뜁니다.");
          delete window[callbackName];
          if (document.querySelector(`script[src*="${callbackName}"]`)) {
             document.querySelector(`script[src*="${callbackName}"]`).remove();
          }
          return; 
        }
        // =================================================================

        // 3. 설정 데이터(rows) 파싱
        posts = (data.rows || []).map((p) => ({
          ID: p.ID || "",
          주제: p.주제 || "",
          키워드: p.키워드 || "",
          발행도메인: parseArray(p.발행도메인),
          백링크: parseArray(p.백링크),
          발행갯수: p.발행갯수 || "", // (예: 40)
          간격_값: p.간격_값 || 30,
          간격_단위: p.간격_단위 || "M",
          상태: p.상태 || "OFF",
          시작시간: p.시작시간 || new Date(new Date().getTime() + 9 * 60 * 60 * 1000).toISOString().slice(0, 19).replace("T", " "),
          발행결과: p.발행결과 || "대기중",
          도메인갯수: p.도메인갯수 || 0,
          백링크갯수: p.백링크갯수 || 0,
          카테고리: p.카테고리 || "",
          태그: p.태그 || "",
            랜덤모드: p.랜덤모드 === "true" || p.랜덤모드 === true, // ✅ 유지 코드 추가
  랜덤개수: parseInt(p.랜덤개수) || 1, // ✅ 유지 코드 추가
          _JobCounter: p._JobCounter || 0, // ✅ 순환 발행 카운터
        }));

        // 4. 로그 데이터(logs) 파싱 및 최신순 정렬
        publishLogs = (data.logs || []).map(log => ({
          time: new Date(log.Time).toLocaleString("ko-KR", { hour12: false }),
          id: log.ID,
          domain: log.Domain,
          topic: log.Topic,
          result: log.Result,
          detail: log.Detail
        })).sort((a, b) => new Date(b.time) - new Date(a.time));

        // 5. 화면 갱신
        renderTable();
        renderPublishLogs();
        renderDomainStats();

        // 6. 콜백 및 스크립트 태그 정리
        delete window[callbackName];
        if (script) script.remove();
      };

      // 7. JSONP 요청
      const script = document.createElement("script");
      script.src = `${SHEET_URL}?callback=${callbackName}&_=${Date.now()}`;
      script.onerror = () => {
         tbody.innerHTML = `<tr><td colspan="18" class="text-red-500 p-4">불러오기 실패: 스크립트 URL을 확인하세요.</td></tr>`;
      };
      document.body.appendChild(script);

    } catch (err) {
      console.error("❌ 시트 불러오기 실패:", err);
      tbody.innerHTML = `<tr><td colspan="18" class="text-red-500 p-4">불러오기 실패: ${err.message}</td></tr>`;
    } finally {
      // ✅ 로드 완료 후 상태 초기화
      isLoading = false;
    }
  }
  /* ===============================
     💾 [STEP 2] 시트 저장하기 (메인)
   =============================== */

  function showLoader(show = true, text = "저장 중...") {
    if (!loader) return;
    loader.classList.toggle("hidden", !show);
    document.getElementById("loader-text").innerText = text;
  }

  async function saveAllRows() {
    showLoader(true); // 로더 표시

    try {
      const iframe = document.createElement("iframe");
      iframe.name = "hiddenFrame";
      iframe.style.display = "none";
      document.body.appendChild(iframe);

const form = document.createElement("form");
form.method = "POST";
form.action = SHEET_URL + "?mode=sync_sheet"; // ✅ 여기에 mode 파라미터 추가!
form.target = "hiddenFrame";

const input = document.createElement("input");
input.type = "hidden";
input.name = "data";
input.value = JSON.stringify({ rows: posts });
form.appendChild(input);
document.body.appendChild(form);
form.submit();
      console.log(`✅ 시트 저장 요청 전송됨 (${posts.length}개 행)`);

      iframe.onload = () => {
        console.log("📬 시트 응답 수신 (iframe)");
        setTimeout(() => {
          iframe.remove();
          form.remove();
        }, 1000);
      };

    } catch (err) {
      console.error("❌ 전체 저장 실패:", err);
    } finally {
      setTimeout(() => showLoader(false), 800);
    }
  }

  /* ===============================
     ➕ [STEP 3] 새 글 추가
   =============================== */
  addRowBtn.addEventListener("click", async () => {
    const now = new Date();
    const kstTime = new Date(now.getTime() + 9 * 60 * 60 * 1000);
    const nowKstFormatted = kstTime.toISOString().slice(0, 19).replace("T", " ");

    const existingIds = posts.map(p => Number(p.ID)).filter(n => !isNaN(n));
    let newId = 1;
    while (existingIds.includes(newId)) newId++;

    const newPost = {
      ID: String(newId),
      주제: "",
      키워드: "",
      발행도메인: [],
      백링크: [],
      발행갯수: "1", // 기본값 '1' (1회 순환)
      간격_값: 30,
      간격_단위: "M",
      상태: "OFF",
      시작시간: nowKstFormatted,
      발행결과: "대기중",
      도메인갯수: 0,
      백링크갯수: 0,
      카테고리: "",
      태그: "",
        랜덤모드: false,
  랜덤개수: 1,
      _JobCounter: 0,
    };

    posts.push(newPost);
    renderTable();

    // 새 행을 "수정" 모드로 변경
    const newIndex = posts.length - 1;
    const row = tbody.rows[newIndex];
    if (!row) return;

    row.querySelectorAll(".editable-cell").forEach(td => td.setAttribute("contenteditable", true));
    row.querySelector(".interval-input").disabled = false;
    row.querySelector(".interval-select").disabled = false;
    row.querySelector(".flatpickr").disabled = false;

    const btn = row.querySelector(".edit-btn");
    if (btn) btn.innerText = "💾 저장";

    await saveAllRows();
  });


/* ===============================
   📋 [STEP 4] 테이블 렌더링 (주제/키워드 관리 추가)
   =============================== */
function renderTable() {
  tbody.innerHTML = "";

  if (!posts.length) {
    tbody.innerHTML = `<tr><td colspan="22" class="p-4 text-gray-400 text-center">데이터 없음</td></tr>`;
    return;
  }

  posts.sort((a, b) => Number(a.ID) - Number(b.ID));

  posts.forEach((p, i) => {
    const row = document.createElement("tr");

    // ✅ 단일 값 호환
    const topics = Array.isArray(p.주제) ? p.주제 : parseArray(p.주제);
    const keywords = Array.isArray(p.키워드) ? p.키워드 : parseArray(p.키워드);

    let resultText = p.발행결과 || "대기중";
    if (resultText.includes("✅ 최종 발행 완료")) row.style.backgroundColor = "#e6fffa";
    else if (resultText.includes("✅")) row.style.backgroundColor = "#f0fff4";
    else if (resultText.includes("❌")) row.style.backgroundColor = "#fff5f5";

    // 랜덤 관련 기본값 유지
    if (p.랜덤모드 === undefined) p.랜덤모드 = false;
    if (p.랜덤개수 === undefined) p.랜덤개수 = 1;

    row.innerHTML = `
      <td class="editable-cell" data-key="ID">${p.ID}</td>
      <td><button class="topic-btn text-blue-600" data-index="${i}">관리</button></td>
      <td>${topics.length}</td>
      <td><button class="keyword-btn text-blue-600" data-index="${i}">관리</button></td>
      <td>${keywords.length}</td>
<td><button class="domain-btn text-blue-600" data-index="${i}">관리</button></td>
<td>${p.도메인갯수}</td>

      <td><button class="link-btn text-blue-600" data-index="${i}">관리</button></td>
      <td>${p.백링크갯수}</td>
      <td>
        <label class="toggle">
          <input type="checkbox" data-index="${i}" class="random-mode-toggle" ${p.랜덤모드 ? "checked" : ""}>
          <span class="slider"></span>
        </label>
      </td>
      <td>
        <input type="number" min="1" value="${p.랜덤개수}" class="random-count-input w-14 text-center border rounded p-1" data-index="${i}" disabled>
      </td>
      <td class="editable-cell" data-key="발행갯수">${p.발행갯수}</td>
            <td><button class="category-btn text-blue-600" data-index="${i}">관리</button></td>
      <td>${p.카테고리 ? p.카테고리.split(",").filter(Boolean).length : 0}</td>
      <td><button class="tag-btn text-blue-600" data-index="${i}">관리</button></td>
      <td>${p.태그 ? p.태그.split(",").filter(Boolean).length : 0}</td>

      <td>
        <input type="number" value="${p.간격_값}" min="1" class="interval-input" data-index="${i}" style="width:60px;" disabled>
        <select class="interval-select" data-index="${i}" disabled>
          <option value="M" ${p.간격_단위 === "M" ? "selected" : ""}>분</option>
          <option value="H" ${p.간격_단위 === "H" ? "selected" : ""}>시</option>
          <option value="D" ${p.간격_단위 === "D" ? "selected" : ""}>일</option>
        </select>
      </td>

      <td>
        <label class="toggle">
          <input type="checkbox" ${p.상태 === "ON" ? "checked" : ""} data-index="${i}">
          <span class="slider"></span>
        </label>
      </td>
      <td><input type="text" class="flatpickr w-full text-center" value="${p.시작시간}" data-id="${p.ID}" disabled></td>
      <td>${resultText}</td>
      <td>
        <button class="edit-btn text-yellow-600" data-index="${i}">✏️ 수정</button>
        <button class="text-red-500 delete-btn" data-index="${i}">🗑 삭제</button>
      </td>
    `;

    tbody.appendChild(row);

    // ✅ flatpickr 적용
    flatpickr(row.querySelector(".flatpickr"), {
      enableTime: true,
      dateFormat: "Y-m-d H:i:S",
      locale: "ko",
      defaultDate: p.시작시간,
      onChange: (dates, str, inst) => {
        const id = inst.element.dataset.id;
        const target = posts.find(x => x.ID === id);
        if (target) target.시작시간 = str;
      },
    });
  });

  attachEvents();
}

  /* ===============================
     ⚙️ [STEP 5] 이벤트 핸들러 부착
   =============================== */
  function attachEvents() {
    
// ---------------------------------
// ✏️ 수정 / 💾 저장 버튼
// ---------------------------------
document.querySelectorAll(".edit-btn").forEach((btn) => {
  btn.onclick = async (e) => {
    const idx = e.target.dataset.index;
    const row = tbody.rows[idx];
    const post = posts[idx];

    if (btn.innerText.includes("수정")) {
      btn.innerText = "💾 저장";
      row.querySelectorAll(".editable-cell").forEach((td) => td.setAttribute("contenteditable", true));
      row.querySelector(".interval-input").disabled = false;
      row.querySelector(".interval-select").disabled = false;
      row.querySelector(".flatpickr").disabled = false;
      // ✅ 수정모드일 때만 랜덤개수 입력 가능
      row.querySelector(".random-count-input").disabled = false;
    } else {
      row.querySelectorAll(".editable-cell").forEach((td) => {
        const key = td.dataset.key;
        post[key] = td.innerText.trim();
        td.removeAttribute("contenteditable");
      });

      post.시작시간 = row.querySelector(".flatpickr").value;
      post.간격_값 = row.querySelector(".interval-input").value;
      post.간격_단위 = row.querySelector(".interval-select").value;
      row.querySelector(".interval-input").disabled = true;
      row.querySelector(".interval-select").disabled = true;
      row.querySelector(".flatpickr").disabled = true;
      // ✅ 저장 후 다시 잠금
      row.querySelector(".random-count-input").disabled = true;

      const rounds = Number(post.발행갯수) || 1;
      const domainCount = post.발행도메인.length > 0 ? post.발행도메인.length : 1;
      post.발행갯수 = rounds * domainCount;
      post._JobCounter = 0;
      row.querySelector("[data-key='발행갯수']").innerText = post.발행갯수;

      post.발행결과 = "대기중";
      post.상태 = "OFF";
      row.cells[16].innerText = "대기중";
      row.querySelector(".toggle input").checked = false;
      row.style.backgroundColor = "";

      btn.innerText = "✏️ 수정";
      await saveAllRows();
    }
  };
});

// ---------------------------------
// 🟢 ON/OFF 토글 → 즉시 발행 트리거 연동
// ---------------------------------
document.querySelectorAll(".toggle input").forEach((chk) => {
  chk.onchange = async (e) => {
    const idx = e.target.dataset.index;
    const post = posts[idx];
    post.상태 = e.target.checked ? "ON" : "OFF";

    if (post.상태 === "ON") {
      post.발행결과 = "🚀 발행 요청 전송 중...";
      if (tbody.rows[idx]) tbody.rows[idx].cells[16].innerText = post.발행결과;
      await saveAllRows();

      // ✅ 주제와 키워드 중 첫 번째 단어만 사용
      const domainKey = (Array.isArray(post.발행도메인) && post.발행도메인.length)
        ? post.발행도메인[0].replace(/^https?:\/\//, "")
        : "xn--on3bn77b.net";

      const keyword = Array.isArray(post.키워드)
        ? post.키워드[0]
        : String(post.키워드 || "테스트 이미지");

      // ✅ Google Apps Script 호출 (autoImagePost 실행)
      fetch(`${SHEET_URL}?action=autoPost&domainKey=${encodeURIComponent(domainKey)}&keyword=${encodeURIComponent(keyword)}&_=${Date.now()}`)

        .then((res) => res.text())
        .then((txt) => {
          console.log("✅ autoImagePost 호출 결과:", txt);
          post.발행결과 = "✅ 발행 요청 완료 (GAS에서 처리 중)";
          if (tbody.rows[idx]) tbody.rows[idx].cells[16].innerText = post.발행결과;
        })
 .catch(async (err) => {
  console.error("❌ autoImagePost 호출 실패:", err);
  post.발행결과 = "❌ GAS 호출 실패";
  if (tbody.rows[idx]) tbody.rows[idx].cells[16].innerText = post.발행결과;
  await saveAllRows(); // 실패 상태도 저장
});
    } else {
      post.발행결과 = "⏸️ 발행 중지됨";
      if (tbody.rows[idx]) tbody.rows[idx].cells[16].innerText = post.발행결과;
      await saveAllRows();
    }
  };
});

    // ---------------------------------
    // 🗑 삭제 버튼
    // ---------------------------------
    document.querySelectorAll(".delete-btn").forEach((btn) => {
      btn.onclick = async (e) => {
        const idx = e.target.dataset.index;
        const post = posts[idx];
        if (!confirm("정말 삭제하시겠습니까? (로그는 유지됩니다)")) return;
        posts.splice(idx, 1);
        renderTable();
        renderDomainStats();
        await saveAllRows();
      };
    });

    // ---------------------------------
    // 🌐 모달 연결
    // ---------------------------------
    document.querySelectorAll(".domain-btn").forEach((b) => (b.onclick = openDomainModal));
    document.querySelectorAll(".link-btn").forEach((b) => (b.onclick = openLinkModal));
    document.querySelectorAll(".category-btn").forEach((b) => (b.onclick = openCategoryModal));
    document.querySelectorAll(".tag-btn").forEach((b) => (b.onclick = openTagModal));

    // ✅ 주제 / 키워드 모달 연결
    document.querySelectorAll(".topic-btn").forEach((b) => (b.onclick = openTopicModal));
    document.querySelectorAll(".keyword-btn").forEach((b) => (b.onclick = openKeywordModal));

    // 🎲 랜덤모드 / 랜덤개수 이벤트
    document.querySelectorAll(".random-mode-toggle").forEach(chk => {
      chk.onchange = async (e) => {
        const idx = e.target.dataset.index;
        posts[idx].랜덤모드 = e.target.checked;
        await saveAllRows();
      };
    });

    document.querySelectorAll(".random-count-input").forEach(inp => {
      inp.onchange = async (e) => {
        const idx = e.target.dataset.index;
        posts[idx].랜덤개수 = parseInt(e.target.value) || 1;
        await saveAllRows();
      };
    });


  } // ✅ attachEvents 끝

  /* ===============================
     🌐 [STEP 6] 모달 관리 (도메인, 백링크) (v3.1 유지)
   =============================== */
  const domainTableBody = document.querySelector("#domainModal tbody");
  const linkTableBody = document.querySelector("#linkModal tbody");
  const addDomainBtn = document.getElementById("addDomain");
  const saveDomainsBtn = document.getElementById("saveDomains");
  const closeDomainModalBtn = document.getElementById("closeDomainModal");
  const addLinkBtn = document.getElementById("addLink");
  const saveLinksBtn = document.getElementById("saveLinks");
  const closeLinkModalBtn = document.getElementById("closeLinkModal");

  function openDomainModal(e) {
    currentPostIndex = e.target.dataset.index;
    domainModal.classList.remove("hidden");
    domainTableBody.innerHTML = "";
    posts[currentPostIndex].발행도메인.forEach((url) => appendDomainRow(url));
  }

  function openLinkModal(e) {
    currentPostIndex = e.target.dataset.index;
    linkModal.classList.remove("hidden");
    linkTableBody.innerHTML = "";
    posts[currentPostIndex].백링크.forEach((l) => appendLinkRow(l));
  }

  function appendDomainRow(url = "") {
    const tr = document.createElement("tr");
    tr.innerHTML = `<td><input type="url" value="${url}" class="w-full border p-1 rounded"></td>
      <td><button class="text-red-500 delete-domain-btn">🗑</button></td>`;
    tr.querySelector(".delete-domain-btn").onclick = () => tr.remove();
    domainTableBody.appendChild(tr);
  }

function appendLinkRow(l = { url: "", anchor: "", option: "본문", fixed: false }) {
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="url" value="${l.url}" class="w-full border p-1 rounded"></td>
    <td><input type="text" value="${l.anchor}" class="w-full border p-1 rounded"></td>
    <td>
      <select class="border p-1 rounded">
        <option value="본문" ${l.option === "본문" ? "selected" : ""}>본문</option>
        <option value="랜덤" ${l.option === "랜덤" ? "selected" : ""}>랜덤</option>
        <option value="첫문단" ${l.option === "첫문단" ? "selected" : ""}>첫문단</option>
        <option value="끝문단" ${l.option === "끝문단" ? "selected" : ""}>끝문단</option>
      </select>
    </td>
    <!-- ✅ 고정 체크박스 -->
    <td class="text-center">
      <input type="checkbox" class="fixed-checkbox" ${l.fixed ? "checked" : ""}>
    </td>
    <td><button class="text-red-500 delete-link-btn">🗑</button></td>
  `;
  tr.querySelector(".delete-link-btn").onclick = () => tr.remove();
  linkTableBody.appendChild(tr);
}


  addDomainBtn.onclick = () => appendDomainRow();
  saveDomainsBtn.onclick = async () => {
    const arr = [];
    domainTableBody.querySelectorAll("input").forEach((i) => {
      if (i.value.trim()) arr.push(i.value.trim());
    });
    posts[currentPostIndex].발행도메인 = arr;
    posts[currentPostIndex].도메인갯수 = arr.length;
    await saveAllRows();
    const row = tbody.rows[currentPostIndex];
    if (row) row.cells[5].innerText = arr.length;
    domainModal.classList.add("hidden");
  };

  addLinkBtn.onclick = () => appendLinkRow();
  saveLinksBtn.onclick = async () => {
  const arr = [];
  linkTableBody.querySelectorAll("tr").forEach((tr) => {
    const inputs = tr.querySelectorAll("input");
    const select = tr.querySelector("select");
    const url = inputs[0].value.trim();
    const anchor = inputs[1].value.trim();
    const fixed = tr.querySelector(".fixed-checkbox").checked;
    if (url && anchor) arr.push({ url, anchor, option: select.value, fixed });
  });
  posts[currentPostIndex].백링크 = arr;
  posts[currentPostIndex].백링크갯수 = arr.length;
  await saveAllRows();
const row = tbody.rows[currentPostIndex];
if (row) row.cells[9].innerText = arr.length;
  linkModal.classList.add("hidden");
};


  closeDomainModalBtn.onclick = () => domainModal.classList.add("hidden");
  closeLinkModalBtn.onclick = () => linkModal.classList.add("hidden");
// ---------------------------------
// 🎲 랜덤 미리보기 버튼
// ---------------------------------
const previewRandomBtn = document.getElementById("previewRandomLinks");
if (previewRandomBtn) {
  previewRandomBtn.onclick = () => {
    if (currentPostIndex === -1) return;
    const post = posts[currentPostIndex];
    const links = post.백링크 || [];
    if (links.length === 0) return alert("백링크가 없습니다.");

    const fixedLinks = links.filter(l => l.fixed);
    const nonFixed = links.filter(l => !l.fixed);
    const count = post.랜덤개수 || 1;

    const randoms = nonFixed.sort(() => 0.5 - Math.random()).slice(0, count);
    const finalSet = [...fixedLinks, ...randoms];

    alert(
      `🎯 랜덤모드: ${post.랜덤모드 ? "ON" : "OFF"}\n` +
      `🔒 고정 ${fixedLinks.length}개 + 🎲 랜덤 ${randoms.length}개 = 총 ${finalSet.length}개\n\n` +
      finalSet.map(l => `${l.anchor} (${l.url})`).join("\n")
    );
  };
}

 /* ===============================
     🏷 [STEP 7] 모달 관리 (카테고리, 태그) (v3.1 유지)
   =============================== */
const categoryModal = document.getElementById("categoryModal");
const categoryTableBody = categoryModal.querySelector("tbody");
const addCategoryBtn = document.getElementById("addCategory");
const saveCategoriesBtn = document.getElementById("saveCategories");
const closeCategoryModalBtn = document.getElementById("closeCategoryModal");

const tagModal = document.getElementById("tagModal");
const tagTableBody = tagModal.querySelector("tbody");
const addTagBtn = document.getElementById("addTag");
const saveTagsBtn = document.getElementById("saveTags");
const closeTagModalBtn = document.getElementById("closeTagModal");

function openCategoryModal(e) {
  currentPostIndex = e.target.dataset.index;
  categoryModal.classList.remove("hidden");
  categoryTableBody.innerHTML = "";
  const cats = (posts[currentPostIndex].카테고리 || "")
    .split(",")
    .map((s) => s.trim())
    .filter(Boolean);
  cats.forEach((c) => appendCategoryRow(c));
}

function openTagModal(e) {
  currentPostIndex = e.target.dataset.index;
  tagModal.classList.remove("hidden");
  tagTableBody.innerHTML = "";
  const tags = (posts[currentPostIndex].태그 || "")
    .split(",")
    .map((s) => s.trim())
    .filter(Boolean);
  tags.forEach((t) => appendTagRow(t));
}

function appendCategoryRow(name = "") {
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="text" value="${name}" class="w-full border p-1 rounded"></td>
    <td><button class="text-red-500 delete-cat-btn">🗑</button></td>
  `;
  tr.querySelector(".delete-cat-btn").onclick = () => tr.remove();
  categoryTableBody.appendChild(tr);
}

function appendTagRow(name = "") {
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="text" value="${name}" class="w-full border p-1 rounded"></td>
    <td><button class="text-red-500 delete-tag-btn">🗑</button></td>
  `;
  tr.querySelector(".delete-tag-btn").onclick = () => tr.remove();
  tagTableBody.appendChild(tr);
}

addCategoryBtn.onclick = () => appendCategoryRow();
addTagBtn.onclick = () => appendTagRow();

saveCategoriesBtn.onclick = async () => {
  const arr = [];
  categoryTableBody.querySelectorAll("input").forEach((i) => {
    if (i.value.trim()) arr.push(i.value.trim());
  });
  posts[currentPostIndex].카테고리 = arr.join(", ");
  const row = tbody.rows[currentPostIndex];
  if (row) row.cells[13].innerText = arr.length; // ✅ 10 → 13
  await saveAllRows();
  categoryModal.classList.add("hidden");
};

saveTagsBtn.onclick = async () => {
  const arr = [];
  tagTableBody.querySelectorAll("input").forEach((i) => {
    if (i.value.trim()) arr.push(i.value.trim());
  });
  posts[currentPostIndex].태그 = arr.join(", ");
  const row = tbody.rows[currentPostIndex];
  if (row) row.cells[15].innerText = arr.length; // ✅ 12 → 15
  await saveAllRows();
  tagModal.classList.add("hidden");
};


closeCategoryModalBtn.onclick = () => categoryModal.classList.add("hidden");
closeTagModalBtn.onclick = () => tagModal.classList.add("hidden");


/* ===============================
     🧠 [STEP 7.1] 주제 & 키워드 관리 모달 (신규 추가)
   =============================== */
const topicModal = document.getElementById("topicModal");
const topicTableBody = topicModal.querySelector("tbody");
const addTopicBtn = document.getElementById("addTopic");
const saveTopicsBtn = document.getElementById("saveTopics");
const closeTopicModalBtn = document.getElementById("closeTopicModal");

const keywordModal = document.getElementById("keywordModal");
const keywordTableBody = keywordModal.querySelector("tbody");
const addKeywordBtn = document.getElementById("addKeyword");
const saveKeywordsBtn = document.getElementById("saveKeywords");
const closeKeywordModalBtn = document.getElementById("closeKeywordModal");

function openTopicModal(e) {
  currentPostIndex = e.target.dataset.index;
  topicModal.classList.remove("hidden");
  topicTableBody.innerHTML = "";
  const arr = parseArray(posts[currentPostIndex].주제);
  arr.forEach((v) => appendTopicRow(v));
}

function openKeywordModal(e) {
  currentPostIndex = e.target.dataset.index;
  keywordModal.classList.remove("hidden");
  keywordTableBody.innerHTML = "";
  const arr = parseArray(posts[currentPostIndex].키워드);
  arr.forEach((v) => appendKeywordRow(v));
}

function appendTopicRow(v = "") {
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="text" value="${v}" class="w-full border p-1 rounded"></td>
    <td><button class="text-red-500 delete-topic-btn">🗑</button></td>
  `;
  tr.querySelector(".delete-topic-btn").onclick = () => tr.remove();
  topicTableBody.appendChild(tr);
}

function appendKeywordRow(v = "") {
  const tr = document.createElement("tr");
  tr.innerHTML = `
    <td><input type="text" value="${v}" class="w-full border p-1 rounded"></td>
    <td><button class="text-red-500 delete-keyword-btn">🗑</button></td>
  `;
  tr.querySelector(".delete-keyword-btn").onclick = () => tr.remove();
  keywordTableBody.appendChild(tr);
}

addTopicBtn.onclick = () => appendTopicRow();
addKeywordBtn.onclick = () => appendKeywordRow();

saveTopicsBtn.onclick = async () => {
  const arr = [];
  topicTableBody.querySelectorAll("input").forEach((i) => {
    if (i.value.trim()) arr.push(i.value.trim());
  });
  posts[currentPostIndex].주제 = arr;
  const row = tbody.rows[currentPostIndex];
  if (row) row.cells[2].innerText = arr.length;
  await saveAllRows();
  topicModal.classList.add("hidden");
};

saveKeywordsBtn.onclick = async () => {
  const arr = [];
  keywordTableBody.querySelectorAll("input").forEach((i) => {
    if (i.value.trim()) arr.push(i.value.trim());
  });
  posts[currentPostIndex].키워드 = arr;
  const row = tbody.rows[currentPostIndex];
  if (row) row.cells[4].innerText = arr.length;
  await saveAllRows();
  keywordModal.classList.add("hidden");
};

closeTopicModalBtn.onclick = () => topicModal.classList.add("hidden");
closeKeywordModalBtn.onclick = () => keywordModal.classList.add("hidden");



/* ===============================
   🧾 [STEP 8] 로그 및 통계 렌더링 (v3.3 확장)
   - 기본 30개만 표시
   - 스크롤 시 최대 60개까지 로드
   - 우측 상단 전체삭제 버튼 추가
   =============================== */
function setupLogTable() {
  const existing = document.getElementById("publishLogSection");
  if (existing) return;

  const logSection = document.createElement("div");
  logSection.id = "publishLogSection";
  logSection.className = "mt-8 bg-gray-50 border rounded-lg p-4 shadow";

  logSection.innerHTML = `
    <div class="flex justify-between items-center mb-2">
      <h2 class="text-lg font-semibold">🧾 발행 로그 (10초마다 자동 갱신)</h2>
      <button id="clearLogsBtn"
        class="px-3 py-1 bg-red-500 text-white rounded hover:bg-red-600 text-sm">
        🗑 전체 삭제
      </button>
    </div>

    <div id="logContainer" style="max-height: 400px; overflow-y: auto;">
      <table id="logTable" class="w-full text-sm border-collapse border border-gray-300">
        <thead class="bg-gray-200">
          <tr>
            <th class="border p-2">시간</th>
            <th class="border p-2">ID</th>
            <th class="border p-2">도메인</th>
            <th class="border p-2">주제</th>
            <th class="border p-2">결과</th>
            <th class="border p-2">상세(링크/오류)</th>
          </tr>
        </thead>
        <tbody class="text-center bg-white"></tbody>
      </table>
    </div>
  `;

  document.body.appendChild(logSection);
  
 // ✅ 전체 로그 삭제 버튼
let autoReload = null;

document.getElementById("clearLogsBtn").addEventListener("click", async () => {
  if (!confirm("정말 전체 로그를 삭제하시겠습니까?")) return;

  try {
    // 🔹 자동 새로고침 완전히 중지
    if (autoReload) clearInterval(autoReload);

    // 🔹 서버(GAS) 로그 완전 삭제 요청
    const url = `${SHEET_URL}?mode=clearLogs&_=${Date.now()}&rand=${Math.random()}`;
    const res = await fetch(url, { method: "GET", cache: "no-store" });
    const text = await res.text();

    if (!text.includes("ok")) {
      alert("⚠️ 서버 로그 삭제 실패 (응답 이상)");
      return;
    }

    // 🔹 클라이언트 즉시 초기화
    publishLogs = [];
    renderPublishLogs();

    alert("🧹 모든 로그가 완전히 삭제되었습니다.");

    // ✅ 1️⃣ 시트 반영 대기
    await new Promise(resolve => setTimeout(resolve, 5000));

    // ✅ 2️⃣ 강제 새로고침
    await loadFromSheet();

    // ✅ 3️⃣ 자동 새로고침 재시작
if (autoReload) clearInterval(autoReload);
setInterval(() => {
  const editing = document.querySelector("[contenteditable=true]");
  const saving = Array.from(document.querySelectorAll(".edit-btn")).some(b => b.innerText.includes("💾"));
  
  // ✅ 로딩 중이 아닐 때만 새로고침 (중복방지)
  if (!editing && !saving && !isLoading) loadFromSheet();
}, 10000);




  } catch (err) {
    alert("❌ 로그 삭제 중 오류 발생: " + err.message);
  }
});

}


function renderPublishLogs() {
  const tbody = document.querySelector("#logTable tbody");
  const container = document.getElementById("logContainer");
  if (!tbody || !container) return;

  // ✅ 최대 60개까지만 유지, 기본은 30개 표시
  const visibleLogs = publishLogs.slice(0, 60);
  tbody.innerHTML = visibleLogs
    .slice(0, 30)
    .map((log) => {
      let detailHtml = log.detail;
      if (log.result === "성공" && log.detail && log.detail.startsWith("http")) {
        detailHtml = `<a href="${log.detail}" target="_blank" class="text-blue-600 hover:underline">링크 열기</a>`;
      } else if (log.result === "실패" && log.detail) {
        detailHtml = `<span title="${log.detail}">${log.detail.substring(0, 30)}...</span>`;
      }

      return `
        <tr>
          <td class="border p-1 text-gray-700">${log.time}</td>
          <td class="border p-1">${log.id}</td>
          <td class="border p-1">${log.domain}</td>
          <td class="border p-1">${log.topic}</td>
          <td class="border p-1 font-semibold ${
            log.result === "성공" ? "text-green-600" : "text-red-600"
          }">${log.result}</td>
          <td class="border p-1 text-xs">${detailHtml}</td>
        </tr>`;
    })
    .join("");

  // ✅ 스크롤 시 추가로 10개씩 로드
  container.onscroll = () => {
    if (container.scrollTop + container.clientHeight >= container.scrollHeight - 10) {
      const currentCount = tbody.children.length;
      const nextLogs = visibleLogs.slice(currentCount, currentCount + 10);
      if (nextLogs.length > 0) {
        tbody.insertAdjacentHTML(
          "beforeend",
          nextLogs
            .map(
              (log) => `
              <tr>
                <td class="border p-1 text-gray-700">${log.time}</td>
                <td class="border p-1">${log.id}</td>
                <td class="border p-1">${log.domain}</td>
                <td class="border p-1">${log.topic}</td>
                <td class="border p-1 font-semibold ${
                  log.result === "성공" ? "text-green-600" : "text-red-600"
                }">${log.result}</td>
                <td class="border p-1 text-xs">${
                  log.detail?.startsWith("http")
                    ? `<a href="${log.detail}" target="_blank" class="text-blue-600 hover:underline">링크 열기</a>`
                    : `<span title="${log.detail}">${log.detail?.substring(0, 30) ?? ""}</span>`
                }</td>
              </tr>`
            )
            .join("")
        );
      }
    }
  };
}


function renderDomainStats() {
  let section = document.getElementById("domainStatsSection");
  if (!section) {
    section = document.createElement("div");
    section.id = "domainStatsSection";
    section.className = "mt-8 bg-white border rounded-lg p-4 shadow";
    document.body.appendChild(section);
  }

  const domainStats = {};
  publishLogs.forEach((log) => {
    if (!log.domain) return;
    if (!domainStats[log.domain]) {
      domainStats[log.domain] = { posts: 0, backlinkDomains: [] };
    }
    if (log.result === "성공") {
      domainStats[log.domain].posts++;
    }

    const post = posts.find((p) => p.ID == log.id);
    if (post && post.백링크 && post.백링크.length) {
      post.백링크.forEach((link) => {
        try {
          const backlinkDomain = new URL(link.url).hostname.replace("www.", "");
          if (!domainStats[log.domain].backlinkDomains.includes(backlinkDomain)) {
            domainStats[log.domain].backlinkDomains.push(backlinkDomain);
          }
        } catch {}
      });
    }
  });

  const rows = Object.entries(domainStats)
    .sort((a, b) => b[1].posts - a[1].posts)
    .map(
      ([domain, stat]) => `
      <tr>
        <td class="border p-1 font-medium">${domain}</td>
        <td class="border p-1 text-center">${stat.posts}</td>
        <td class="border p-1 text-center">${stat.backlinkDomains.length}</td>
        <td class="border p-1 text-blue-600 cursor-pointer hover:underline"
            onclick="showDomainDetail('${domain}')">
          상세보기
        </td>
      </tr>`
    )
    .join("");

  section.innerHTML = `
    <h2 class="text-lg font-semibold mb-2">🌐 도메인별 발행 현황 (실시간)</h2>
    <table class="w-full text-sm border-collapse border border-gray-300">
      <thead class="bg-gray-200">
        <tr>
          <th class="border p-2">도메인</th>
          <th class="border p-2">발행 성공 수</th>
          <th class="border p-2">연결된 백링크 도메인 수</th>
          <th class="border p-2">상세</th>
        </tr>
      </thead>
      <tbody class="text-center bg-white">
        ${rows || '<tr><td colspan="4" class="p-2 text-gray-400">아직 데이터 없음</td></tr>'}
      </tbody>
    </table>
  `;
}

  
  /* ===============================
     🌐 [STEP 9] 도메인 상세보기 모달 (v3.1 유지)
   =============================== */
   
  function setupDomainModal() {
    let modal = document.getElementById("domainDetailModal");
    if (!modal) {
      modal = document.createElement("div");
      modal.id = "domainDetailModal";
      modal.className = "hidden fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50";
      modal.innerHTML = `
        <div class="bg-white rounded-xl shadow-2xl p-6 w-full max-w-2xl relative">
          <h2 class="text-xl font-bold mb-3 text-gray-700 border-b pb-2" id="domainDetailTitle">도메인 상세보기</h2>
          <div id="domainDetailBody" class="max-h-80 overflow-y-auto text-sm text-gray-700"></div>
          <button id="closeDomainDetail" class="mt-5 px-4 py-2 bg-gray-700 text-white rounded-lg hover:bg-gray-800 w-full font-semibold">
            닫기
          </button>
        </div>
      `;
      document.body.appendChild(modal);
      document.getElementById("closeDomainDetail").onclick = () => modal.classList.add("hidden");
    }
  }

  /**
   * (상세보기) 클릭 시 모달 표시
   */
  window.showDomainDetail = (domain) => { 
    const modal = document.getElementById("domainDetailModal");
    const title = document.getElementById("domainDetailTitle");
    const body = document.getElementById("domainDetailBody");
    if (!modal || !title || !body) return;

    const successLogs = publishLogs.filter(log => log.domain === domain && log.result === "성공");
    
    let allBacklinkDomains = new Set();
    successLogs.forEach(log => {
        const post = posts.find(p => p.ID == log.id);
         if (post && post.백링크 && post.백링크.length) {
            post.백링크.forEach(link => {
                try {
                    const backlinkDomain = new URL(link.url).hostname.replace("www.", "");
                    allBacklinkDomains.add(backlinkDomain);
                } catch {}
            });
        }
    });
    
    const backlinks = Array.from(allBacklinkDomains);
    const lastLog = successLogs[0];
    const lastTime = lastLog ? lastLog.time : "데이터 없음";

    title.innerHTML = `
      🌐 ${domain}
      <div class="text-sm text-gray-500 mt-1 font-normal">
        📑 발행 횟수: <b class="text-blue-600">${successLogs.length}</b>회 |
        🕒 마지막 발행: <b class="text-green-700">${lastTime}</b> |
        🔗 백링크 도메인: <b class="text-purple-600">${backlinks.length}</b>개
      </div>
    `;

    if (backlinks.length === 0) {
      body.innerHTML = `<p class="text-gray-500 text-center py-4">연결된 백링크 도메인 없음</p>`;
    } else {
      body.innerHTML = `
        <table class="w-full text-left border-collapse border border-gray-300">
          <thead class="bg-gray-100">
            <tr>
              <th class="border p-2 w-16 text-center">#</th>
              <th class="border p-2">백링크 도메인</th>
            </tr>
          </thead>
          <tbody>
            ${backlinks.map((d, i) => `
              <tr>
                <td class="border p-1 text-center">${i + 1}</td>
                <td class="border p-1 text-blue-600 underline cursor-pointer" onclick="window.open('https://${d}', '_blank')">${d}</td>
              </tr>
            `).join("")}
          </tbody>
        </table>
      `;
    }

    modal.classList.remove("hidden");
  }


  /* ===============================
     🚀 [STEP 10] 스크립트 실행
   =============================== */
  
  // 1. 하단 테이블/모달 DOM 생성
  setupLogTable();
  setupDomainModal();
  
  // 2. 최초 데이터 로드
  loadFromSheet(); 

  // 3. 10초마다 자동 새로고침 (단, 수정 중일 때는 제외)
  // ✅ 자동 새로고침 (수정 중일 때 덮어쓰기 방지)
setInterval(() => {
  const editing = document.querySelector("[contenteditable=true]");
  const saving = Array.from(document.querySelectorAll(".edit-btn")).some(b => b.innerText.includes("💾"));
  
  // ✅ 로드 중이거나 마지막 업데이트 직후에는 새로고침 금지
  const now = Date.now();
  if (!window.lastLoadTime) window.lastLoadTime = 0;

  // ✅ 최소 12초 이상 간격을 유지 (중복 갱신 차단)
  if (!editing && !saving && !isLoading && now - window.lastLoadTime > 12000) {
    window.lastLoadTime = now;
    loadFromSheet();
  }
}, 10000);


});